################################################################
# Inclusion of time-varying covariates in cure survival models #
# with application in fertility studies                        #
#                                                              #
# by P.Lambert & V. Bremhorst                                  #
#                                                              #
# R Code of the application section.                           #
################################################################

# Cleaning the workspace.
remove(list = ls())

# Working directory
## setwd("TO_COMPLETE_WITH_YOUR_DIRECTORY)

# Needed packages 
library(coda) 
library(compiler) 
library(survival)

# Fortran routines (only on Linux)
dyn.load("Lambert_Bremhorst.so")

# Fix the seed.
set.seed(1)

# Define all the constants used during the code.

#1. Number Of knots to contruct the splines
nknots_BD <- 17
degree_BD <- 3
order_BD <- 3
n_splines_BD <- nknots_BD + degree_BD

#2. Roughness penalty prior distribution hyper-parameters 
nu_BD = 2
adelta_BD <- 0.0001
bdelta_BD <- 0.0001

#3. Hyper-parameters : prior distribution frailty & regression parameters
sd_Cov <- 10
sd_frailty <- 2.5

#4. Iteration and length of the burnin period 
iteration = 150000
updatesd = 50000


#5. Initial SD proposition for the proposal distribution (cf. adaptive Metropolis steps)
phi_BD_Init_SD_prop <- 0.1
beta_Init_SD_prop <- 0.1
lambda_Init_SD_prop <- 0.1
frailty1_Init_SD_prop <- 0.1
frailty2_Init_SD_prop <- 0.1



# Import the data.
data <- read.table(file = "SecondBirth.txt", header = TRUE, sep = "\t") # Second Birth dataset
#data <- read.table(file = "ThirdBirth.txt", header = TRUE, sep = "\t") # Third Birth dataset

# Define the event indicator (as a binary variables)
data$EVENT <- relevel(data$EVENT, ref = "No Birth") 
data$EVENT <- model.matrix(~data$EVENT)[,2]
EVENT <- data$EVENT

# All covariates enter both regression models ==> restrict to the classical Cox Model 
END <- data$END - data$BEGIN 
BEGIN <- rep(0, length(END)) # Always "a new start"

# Sample size & number of observations per subject
number_of_record <- dim(data)[1]
number_of_measurement_Psubject <- table(data$ID) 
n <- length(number_of_measurement_Psubject) 
CNOMPS <- cumsum(c(0, number_of_measurement_Psubject))

# Data at baseline... Needed to center the baseline continuous covariates.
data_baseline <- data[CNOMPS[-length(CNOMPS)] +1, ]

#1. Standardisation of Age of the mother at first birth.
AGEKID1C <- ( data_baseline$AGEKID1C - mean( data_baseline$AGEKID1C ) ) / sd (data_baseline$AGEKID1C) 
AGEKID1C <- rep(AGEKID1C, number_of_measurement_Psubject) 

#2. Standardisation of Age of the calendar period.
PERIOD <- (data_baseline$PERIOD - mean(data_baseline$PERIOD) ) / sd(data_baseline$PERIOD)
PERIOD <- rep(PERIOD, number_of_measurement_Psubject)

#3. Time elapsed between the two first chidren. (Only for third birth)
#SPACE <- ( data_baseline$SPACE_C - mean(data_baseline$SPACE_C) ) / sd( data_baseline$SPACE_C )
#SPACE <- rep(SPACE, number_of_measurement_Psubject)

# Categorical variable into binary variables.
#1. Mother's education level
EDU02 <- data$EDU02

EDU_recode <- c()
EDU_recode[EDU02=="Vocational degree"] <- 0 # Vocational degree is the reference!

ND <- EDU_recode
ND[EDU02 == "no degree"] <- 1
ND[EDU02 == "Uni (or FH) degree"] <- 0

UD <- EDU_recode
UD[EDU02 == "Uni (or FH) degree"] <- 1
UD[EDU02 == "no degree"] <- 0

#2. Partner's education level
PEDU02 <- data$PEDU02

PEDU_recode <- c()
PEDU_recode[PEDU02=="Vocational degree"] <- 0 # Vocational degree is the reference!

PND <- PEDU_recode
PND[PEDU02 == "no degree"] <- 1
PND[PEDU02 == "Uni (or FH) degree"] <- 0
PND[PEDU02 == "no partner"] <- 0

PUD <- PEDU_recode
PUD[PEDU02 == "Uni (or FH) degree"] <- 1
PUD[PEDU02 == "no degree"] <- 0
PUD[PEDU02 == "no partner"] <- 0

NP <- PEDU_recode
NP[PEDU02 == "no partner"] <- 1
NP[PEDU02 == "no degree"] <- 0
NP[PEDU02 == "Uni (or FH) degree"] <- 0

# 3. Sex composition of prior children. (Only for third birth)
#SEXCOMP <- data$SEXCOMP

#SEXCOMP_recode <- c()
#SEXCOMP_recode[SEXCOMP == "mix"] <- 0 # Mixed is the reference

#BB <- SEXCOMP_recode
#BB[SEXCOMP == "boys"] <- 1
#BB[SEXCOMP =="girls"] <- 0

#BG <- SEXCOMP_recode
#BG[SEXCOMP == "girls"] <- 1
#BG[SEXCOMP == "boys"] <- 0

# Design matrix X (probability model) 
X <- cbind(1, ND, UD, PND, PUD, NP, PERIOD, AGEKID1C)
#X <- cbind(1, BB, BG, ND, UD, PND, PUD, NP, PERIOD, AGEKID1C, SPACE) # For third birth

nbeta <- dim(X)[2]

# Design matrix W (timing model) 
W <- X[,-c(1)]
nlambda <- dim(W)[2]

# Lower bound of the grid.
xl <- 0
# Upper bound of the grid.
xr <- max(END) 

# Definiton of the bins and of the middle points of the bins.
bins <- 300
partition <- seq(xl, xr, length = bins+1)
width <- partition[2] - partition[1]
middleBins <- partition[1:bins] + (width/2)

# Tools to evaluate the survival function using the rectangle method.
upto <- as.integer(END/width) + 1
upto[which(upto == bins +1)] <- bins

# Maximum event time (in covariate configuration)
max_END <- max(END[EVENT == 1])

# Check how many spline coefficient need to be estimated.
L <- (xr-xl) / (n_splines_BD) 
n_splines_estimate <- min(ceiling(max_END / L), n_splines_BD - 1)  
rank_penalty <- min(n_splines_BD, n_splines_estimate + 2)


# Definition of the B-splines basis. 
Bmiddle <- matrix(.Fortran("cubicBsplines", 
                     x = as.double(middleBins), 
                     nx = as.integer(bins), 
                     xl = as.double(xl), 
                     xr = as.double(xr), 
                     ndx = as.integer(nknots_BD), 
                     B = as.double(matrix(0, nrow = bins, ncol = n_splines_BD)))$B, ncol = n_splines_BD)

Bobs <- matrix(.Fortran("cubicBsplines", 
                     x = as.double(END), 
                     nx = as.integer(number_of_record), 
                     xl = as.double(xl), 
                     xr = as.double(xr), 
                     ndx = as.integer(nknots_BD), 
                     B = as.double(matrix(0, nrow = number_of_record, ncol = n_splines_BD)))$B, ncol = n_splines_BD)

# Roughness penalty : Difference Matrix.
penalty_BD <- matrix(.Fortran("penmat",
                   nknots = as.integer(rank_penalty - degree_BD), 
                   degree = as.integer(degree_BD),
                   order = as.integer(order_BD), 
                   res = as.double(matrix(0, ncol = rank_penalty, nrow = rank_penalty)))$res, ncol = rank_penalty)
penalty_BD <- penalty_BD + diag(rep(1, rank_penalty))*10^(-6)


# Tools to defined the parameters in the log-posterior function below.
index_phi_BD <- 1:n_splines_estimate
index_beta <- (index_phi_BD[length(index_phi_BD)] + 1):(index_phi_BD[length(index_phi_BD)] + nbeta)
index_lambda <- (index_beta[length(index_beta)] + 1):(index_beta[length(index_beta)] + nlambda)
index_pen_BD <- (index_lambda[length(index_lambda)] + 1):(index_lambda[length(index_lambda)] + 1)
index_delta_BD <- (index_pen_BD[length(index_pen_BD)] + 1):(index_pen_BD[length(index_pen_BD)] + 1)
index_frailty1 <- (index_delta_BD[length(index_delta_BD)] + 1):(index_delta_BD[length(index_delta_BD)] + 1)
index_frailty2 <- (index_frailty1[length(index_frailty1)] + 1):(index_frailty1[length(index_frailty1)] + 1)   

# Log posterior function
log.post <- function(param) { 
  
   #  Parameters to estimate 
#_____________________________________________________________

   phi_BD <- c(param[index_phi_BD], rep(10, n_splines_BD - n_splines_estimate)) # Spline parameters 

   beta <- param[index_beta] # Regression parameters in theta

   lambda <- param[index_lambda] # Regression parameters in the Cox Model
   
   pen_BD <- exp( param[ index_pen_BD ] ) # penalty parameter

   delta_BD <- exp( param[ index_delta_BD ] ) 
   # parameter of the frailty distribution 

   # frailty1 = xi in the PVF expression ( xi > 0 ) 
   # With this parametrisation frailty1 can take any real value ==> Classiscal Mstep is OK
   frailty1 <- exp( param[ index_frailty1 ] ) # parameter of the frailty distribution 
    
   # frailty2 = nu in the PVF expression (0 < nu < 1) 
   # With this parametrisation frailty2 can take any real value ==> Classiscal Mstep is OK
   frailty2 <- exp( param[ index_frailty2 ] ) / ( 1 + exp( param[ index_frailty2 ] ) )

   if(frailty2 < 0.001) { frailty2 <- 0.001 }
   else if (frailty2 > 0.999) { frailty2 <- 0.999}

   # frailty3 = mu in the PVF expression ( mu  > 0 ) 
   # Fixed to 1
   frailty3 <- 1 

   
   # Quantities for evaluation the likelihood 
#_____________________________________________________________

   baseline.hazard <- exp(Bmiddle%*%phi_BD)
   cum.baseline.hazard <- cumsum(baseline.hazard)*width
   
   Cov.Cox <- exp(W%*%lambda) 
   latent.hazard <- exp((Bobs%*%phi_BD))*Cov.Cox
   
   theta <- exp(X%*%beta)

   # The likelihood 
#_____________________________________________________________


   index.theta <- 1 
   index.Cox <- 1 

   latent.survival <- rep(0, dim(X)[1]) 
   latent.density <- c() 
   logLaplace <- c()
   logD1Laplace <- c() 
   EVENT_Last_record <- c() 

   for( i in 1:n ) {
      
      tools <- 0
      j <- 1 

      while( j <= number_of_measurement_Psubject[i] ) {
      
         latent.survival[index.theta] <- latent.survival[index.theta] +
                                         Cov.Cox[index.Cox] * cum.baseline.hazard[upto[index.Cox]] 
         j <- j + 1

         while( index.Cox+1 <= dim(W)[1] & BEGIN[index.Cox+1] != 0 ) {
            
            index.Cox <- index.Cox + 1

            latent.survival[index.theta] <- latent.survival[index.theta] +
                       Cov.Cox[index.Cox] * ( cum.baseline.hazard[upto[index.Cox]] - cum.baseline.hazard[upto[index.Cox-1]] )

            j <- j + 1 

         }

         latent.survival[index.theta] <- exp( - latent.survival[index.theta] )

         latent.density[index.theta] <- latent.hazard[index.Cox] * latent.survival[index.theta] 
         if(is.na(latent.density[index.theta]) | latent.density[index.theta] < 0.0000001) { latent.density[index.theta] <- 0.0000001 } # avoid NaN (numerical issue)

         tools <- tools + ( theta[ index.theta ] *  ( 1 - latent.survival[ index.theta ] ) )
         index.theta <- index.theta + 1 
         index.Cox <- index.Cox + 1

      } 


   EVENT_Last_record[i] <- EVENT[ index.Cox - 1 ]
   
   logLaplace[i] <- frailty2 / ( frailty1*(1-frailty2) ) * 
                    ( 1 - ( 1 + ( frailty1 * frailty3 * tools / frailty2 ) ) ** (1-frailty2) )  

   logD1Laplace[i] <-  log(frailty3) + 
                       log( theta[ index.theta - 1 ] ) + log( latent.density[index.theta - 1 ] ) -
                       frailty2 * log( 1 + (frailty1*frailty3*tools/frailty2 ) ) +
                       logLaplace[i]

   }

   loglik <- sum( (1-EVENT_Last_record)*logLaplace + EVENT_Last_record * logD1Laplace ) 

   # Prior distribution of the spline parameters 
#_____________________________________________________________

   t1 =  rank_penalty * log(pen_BD) / 2
   t2 = -pen_BD * phi_BD[1:rank_penalty]%*%penalty_BD%*%phi_BD[1:rank_penalty] / 2

   # Prior distribution of the roughness penalty 
#_____________________________________________________________

   t3 = ( (nu_BD/2) - 1) * log(pen_BD)
   t4 = -(nu_BD*delta_BD/2) * pen_BD
   t5 = (nu_BD/2) * log(nu_BD*delta_BD/2)

   t6 = (adelta_BD - 1) * log(delta_BD) 
   t7 = -bdelta_BD * delta_BD

   # Prior distribution of the regression parameters 
#_____________________________________________________________

   t8 = 0
   for(i in 1:nbeta) {
      t8 = t8 - beta[i]**2 / (2*sd_Cov**2)
   }

   for(i in 1:nlambda) {
      t8 = t8 - lambda[i]**2 / (2*sd_Cov**2)
   }

   # Prior distribution of the parameter of the frailty distribution 
#______________________________________________________________________

   t9 =  -param[length(param)-1]**2 / (2*sd_frailty**2)
   t10 =  -param[length(param)]**2 / (2*sd_frailty**2)

   return(-(loglik+t1 + t2 + t3 + t4 + t5 + t6 + t7 + t8 + t9 + t10))

}

# Speed up the computation.
log.post.c <- cmpfun(log.post)

param <- rep(0, index_frailty2)
cat("\n \n Frequentist optimization is running.... please wait... \n \n")
op <- nlm(log.post.c, param, iterlim = 250, hessian = TRUE)
cat("Frequentist optimization done \n \n")

# Estimated correlation structures.
sigma <- solve(op$hessian)
sigma.phiBD <- sigma[index_phi_BD,index_phi_BD]
sigma.beta <- sigma[index_beta, index_beta]
sigma.lambda <- sigma[index_lambda, index_lambda]

# Posterior model as the starting point of the MCMC algorithm.
phi_BD_Init <- c(op$estimate[index_phi_BD],rep(10, n_splines_BD - n_splines_estimate))
beta_Init <- op$estimate[index_beta]
lambda_Init <- op$estimate[index_lambda]
pen_BD_Init <- exp(op$estimate[index_pen_BD])
delta_BD_Init <- exp(op$estimate[index_delta_BD])
frailty1_Init <- op$estimate[ index_frailty1 ]
frailty2_Init <- op$estimate[ index_frailty2 ]

cat("Posterior sampling is running.... please wait... \n \n")

# This function is one of the imported fortran routines.
res = .Fortran("PromTimeCTD", 

        iteration = as.integer(iteration), 
        updatesd = as.integer(updatesd), 

        number_of_subject = as.integer(n),
        number_of_record_theta = as.integer(dim(X)[1]), 
        number_of_record_Cox = as.integer(dim(W)[1]), 
        number_of_measurement_Psubject = as.integer(number_of_measurement_Psubject), 

        Event = as.double(EVENT), 
        BEGIN = as.double(BEGIN), 


        Nsplines_BD = as.integer(n_splines_BD), 
        n_splines_estimate = as.integer(n_splines_estimate), 
        rank_penalty = as.integer(rank_penalty), 

        phi_BD = as.double(matrix(0, nrow = iteration, ncol = n_splines_BD)), 
        phi_BD_Init = as.double(phi_BD_Init), 
        SB_BD = as.double(Bobs), 
        SBM_BD = as.double(Bmiddle), 
        Nbins = as.integer(bins), 
        width = as.double(width), 
        upto = as.integer(upto), 

        penalty_BD = as.double(penalty_BD), 
        pen_BD = as.double(rep(0, iteration+1)),
        pen_BD_Init = as.double(pen_BD_Init), 

        nu_BD = as.double(nu_BD), 
        delta_BD = as.double(rep(0, iteration+1)), 
        delta_BD_Init = as.double(delta_BD_Init), 
        aDelta_BD = as.double(adelta_BD), 
        bDelta_BD = as.double(bdelta_BD),  

        frailty1 = as.double(rep(0, iteration)), 
        frailty1_Init = as.double(frailty1_Init),

        frailty2 = as.double(rep(0, iteration)), 
        frailty2_Init = as.double(frailty2_Init),

        X = as.double(X), 
        beta = as.double(matrix(0, nrow = iteration, ncol = nbeta)), 
        beta_Init = as.double(beta_Init), 
        nbeta = as.integer(nbeta), 
 
        W = as.double(W), 
        lambda = as.double(matrix(0, nrow = iteration, ncol = nlambda)), 
        lambda_Init = as.double(lambda_Init), 
        nlambda = as.integer(nlambda), 

        sd_Cov = as.double(sd_Cov), 
        sd_frailty = as.double(sd_frailty), 

        phi_BD_Init_SD_prop = as.double(phi_BD_Init_SD_prop), 
        beta_Init_SD_prop = as.double(beta_Init_SD_prop), 
        lambda_Init_SD_prop = as.double(lambda_Init_SD_prop), 
        frailty1_Init_SD_prop = as.double(frailty1_Init_SD_prop), 
        frailty2_Init_SD_prop = as.double(frailty2_Init_SD_prop), 

        accept = as.integer(matrix(0, nrow = iteration, ncol = 5)), 
  
        sigma_phi_BD = as.double(sigma.phiBD), 
        sigma_beta = as.double(sigma.beta), 
        sigma_lambda = as.double(sigma.lambda))

cat("Posterior sampling done. \n \n")

tools <- matrix(res$accept, ncol = 5)[-c(1:updatesd), ]
cat("acceptance Rate (after burnin) : " , apply(tools, 2, mean), "\n", "--- \n") 

# Extract the posterior samples
phiPost_BD <- matrix(res$phi_BD, ncol = n_splines_BD)[-c(1:updatesd), ]
betaPost <- matrix(res$beta, ncol = nbeta)[-c(1:updatesd), ]
lambdaPost <- matrix(res$lambda, ncol = nlambda)[-c(1:updatesd), ]
penPost_BD <- res$pen_BD[-c(1:(updatesd+1))]
deltaPost_BD <- res$delta_BD[-c(1:(updatesd+1+1))]
frailty1Post <- res$frailty1[-c(1:(updatesd+1))] 
frailty2Post <- res$frailty2[-c(1:(updatesd+1))]

# To reduce the computation time of the hazard function... we tinned the chain with 1/10 rate.
phiPost10_BD <- matrix(0, nrow = (iteration-updatesd)/10, ncol = n_splines_BD)
betaPost10 <- matrix(0, nrow = (iteration-updatesd)/10, ncol = nbeta)
lambdaPost10 <- matrix(0, nrow = (iteration-updatesd)/10, ncol = nlambda)

penPost10_BD <- c()
deltaPost10_BD <- c()
frailty1Post10 <- c() 
frailty2Post10 <- c() 

for(i in 1:(iteration-updatesd)/10) {

    phiPost10_BD[i,] = phiPost_BD[((i-1)*10) +1,]
    betaPost10[i,] = betaPost[((i-1)*10) +1,]
    lambdaPost10[i,] = lambdaPost[((i-1)*10) +1,]  
    penPost10_BD[i] = penPost_BD[((i-1)*10) +1]
    deltaPost10_BD[i] = deltaPost_BD[((i-1)*10) +1]
    frailty1Post10[i] = frailty1Post[((i-1)*10) +1]
    frailty2Post10[i] = frailty2Post[((i-1)*10) +1]

}

# Geweke convergence diagnostic.
geweke <- c( geweke.diag(betaPost10)$z, 
             geweke.diag(lambdaPost10)$z, 
             geweke.diag(phiPost10_BD[,1:n_splines_estimate])$z, 
             geweke.diag(penPost10_BD)$z, 
             geweke.diag(deltaPost10_BD)$z, 
             geweke.diag(frailty1Post10)$z, 
             geweke.diag(frailty2Post10)$z)


# Trace and ACF plot of the posterior chains.

# 1. Splines coefficients.
jpegname <- paste( "TraceBaseline", "jpeg", sep = ".")
jpeg(jpegname) 
par(mfrow = c(4,5)) 
for(i in 1:n_splines_estimate) {
   plot(phiPost10_BD[,i], type = "l", ylab = "" ) 
}
plot(penPost10_BD, type = "l", ylab = expression(tau)) 
plot(deltaPost10_BD, type = "l", ylab = expression(delta)) 
dev.off() 

jpegname <- paste( "ACFBaseline", "jpeg", sep = ".")
jpeg(jpegname) 
par(mfrow = c(4,5)) 
for(i in 1:n_splines_estimate) {
   acf(phiPost10_BD[,i], main = "" ) 
}
acf(penPost10_BD, main = expression(tau)) 
acf(deltaPost10_BD, main = expression(delta))
dev.off() 

# 2. Regression parameter (probability model).
jpegname <- paste( "TraceREGCURE", "jpeg", sep = ".")
jpeg(jpegname) 
par(mfrow = c(4,4))  
for(i in 1:nbeta) {
  plot(betaPost10[,i], type = "l", ylab = "")
}
dev.off()

jpegname <- paste( "ACFREGCURE", "jpeg", sep = ".")
jpeg(jpegname) 
par(mfrow = c(4,4))  
for(i in 1:nbeta) {
  acf(betaPost10[,i], main = "")
}
dev.off()

# 2. Regression parameter (timing model).
jpegname <- paste( "TraceREGCOX", "jpeg", sep = ".")
jpeg(jpegname) 
par(mfrow = c(4,4))  
for(i in 1:nlambda) {
   plot(lambdaPost10[,i], type = "l", ylab = "")   
}
dev.off()

jpegname <- paste( "ACFREGCOX", "jpeg", sep = ".")
jpeg(jpegname) 
par(mfrow = c(4,4))  
for(i in 1:nlambda) {
   acf(lambdaPost10[,i], main = "")   
}
dev.off()

# Numerical summary of the posterior chains

betaHPD1 <- HPDinterval(mcmc(betaPost10), prob = 0.90)
betaHPD2 <- HPDinterval(mcmc(betaPost10), prob = 0.95)
betaHPD3 <- HPDinterval(mcmc(betaPost10), prob = 0.99)

lambdaHPD1 <- HPDinterval(mcmc(lambdaPost10), prob = 0.90)
lambdaHPD2 <- HPDinterval(mcmc(lambdaPost10), prob = 0.95)
lambdaHPD3 <- HPDinterval(mcmc(lambdaPost10), prob = 0.99)

frailty1HPD1 <- HPDinterval(mcmc(exp(frailty1Post10)), prob = 0.90)
frailty1HPD2 <- HPDinterval(mcmc(exp(frailty1Post10)), prob = 0.95)
frailty1HPD3 <- HPDinterval(mcmc(exp(frailty1Post10)), prob = 0.99)

frailty2HPD1 <- HPDinterval(mcmc(exp(frailty2Post10) / ( 1 + exp(frailty2Post10) )), prob = 0.90)
frailty2HPD2 <- HPDinterval(mcmc(exp(frailty2Post10) / ( 1 + exp(frailty2Post10) )), prob = 0.95)
frailty2HPD3 <- HPDinterval(mcmc(exp(frailty2Post10) / ( 1 + exp(frailty2Post10) )), prob = 0.99)

Cnames <- c("Estimation", "post sd", "Lower90", "Upper90", "Lower95",  "Upper95", "Lower99", "Upper99")
result <- matrix(nrow = nbeta+nlambda, ncol = length(Cnames)) 
dimnames(result) <- list(c("Intercept", colnames(X)[-c(1)],  colnames(W)), Cnames)
result[,1] <- c(apply(betaPost10, 2, median), apply(lambdaPost10, 2, median))
 
result[,2] <- c(apply(betaPost10, 2, sd), apply(lambdaPost10,2,sd))

result[,3] <- c(betaHPD1[,1], lambdaHPD1[,1])
result[,4] <- c(betaHPD1[,2], lambdaHPD1[,2])

result[,5] <- c(betaHPD2[,1], lambdaHPD2[,1])
result[,6] <- c(betaHPD2[,2], lambdaHPD2[,2])

result[,7] <- c(betaHPD3[,1], lambdaHPD3[,1])
result[,8] <- c(betaHPD3[,2], lambdaHPD3[,2])

sink("result.txt")
print(round(result,3))
sink()


# Computation of the hazard and survival functions. 
Q90 <- quantile(END[EVENT == 1], p = 0.9) # Right queue discarded due to huge uncertainty.

XXS <- seq(xl, xr, by = 0.5)
midXXS <- XXS[1:(length(XXS)-1)] + (XXS[2]-XXS[1])/2
widthXXS <- XXS[2]-XXS[1]

BXXS <- matrix(.Fortran("cubicBsplines", 
               x = as.double(XXS[-1]), 
               nx = as.integer(length(XXS[-1])), 
               xl = as.double(xl), 
               xr = as.double(xr), 
               ndx = as.integer(nknots_BD), 
               B = as.double(matrix(0, nrow = length(XXS[-1]), ncol = n_splines_BD)))$B, ncol = n_splines_BD)

BmidXXS <- matrix(.Fortran("cubicBsplines", 
               x = as.double(midXXS), 
               nx = as.integer(length(midXXS)), 
               xl = as.double(xl), 
               xr = as.double(xr), 
               ndx = as.integer(nknots_BD), 
               B = as.double(matrix(0, nrow = length(midXXS), ncol = n_splines_BD)))$B, ncol = n_splines_BD)


# Baseline hazard
hazardEst <- exp(BXXS %*% t(phiPost10_BD)) 

# Baseline survival
temp <- exp(BmidXXS %*% t(phiPost10_BD)) 
tempCum <- apply(temp, 2, cumsum)*widthXXS 
survivalEst <- exp(-tempCum)

survivalEstMed <- apply(survivalEst, 1, median)
survivalEst025 <- apply(survivalEst, 1, quantile, 0.025)
survivalEst975<- apply(survivalEst, 1, quantile, 0.975) 

png("Second birth-Survival95.png")
#png("Third birth-Survival95.png")
plot(XXS[-1], survivalEstMed, type = "n", xlab = "time (in months)", ylab = "", main = "Baseline survival function")
polygon(c(XXS[-1], rev(XXS[-1])), c(survivalEst025, rev(survivalEst975)), border = F, col = "grey")
lines(XXS[-1], survivalEstMed, col = 1, lty = 2, lwd = 2)
dev.off()
